# Stock Management Architecture Changes

## Overview
Changed from manual product stock management to automatic calculation based on raw material availability.

## Database Changes

### ✅ Removed Columns from `products` Table:
- `current_stock` - No longer needed, calculated dynamically
- `min_stock_level` - No longer needed, calculated dynamically

### Migration:
```bash
php artisan make:migration remove_stock_columns_from_products_table
php artisan migrate
```

### Current Products Table Structure:
```
- id
- name
- description
- sku (unique)
- price
- is_active
- created_at
- updated_at
```

## Code Changes

### 1. Product Model (`app/Models/Product.php`)
**Removed from fillable:**
- `current_stock`
- `min_stock_level`

**Removed from casts:**
- `current_stock`
- `min_stock_level`

**Added Methods:**
```php
public function calculateCanMake(): int
public function canMake(): bool
```

### 2. Product Controller (`app/Http/Controllers/ProductController.php`)
**Removed from validation:**
- `current_stock`
- `min_stock_level`

**Removed from create/update:**
- No longer saves stock values

### 3. Views Updated:
- `resources/views/products/create.blade.php` - Removed stock fields
- `resources/views/products/edit.blade.php` - Removed stock fields, shows calculated capacity
- `resources/views/products/index.blade.php` - Shows "Can Make" instead of stock
- `resources/views/inventory/products.blade.php` - Shows calculated values

### 4. Inventory Service (`app/Services/InventoryService.php`)
**Removed Method:**
- `getLowStockProducts()`

**Added Method:**
- `getUnavailableProducts()` - Returns products that can't be made

## How It Works Now

### Product Stock Calculation:
```
For each product:
  1. Get all required raw materials
  2. For each raw material:
     - Calculate: available_stock / quantity_required
  3. Take the MINIMUM value across all raw materials
  4. That's how many units can be made
```

### Example:
```
Product: Moisturizing Lotion
Requires:
  - 0.25L Lotion Base (Available: 10L) → Can make 40 bottles
  - 1 Bottle (Available: 30 pieces) → Can make 30 bottles  
  - 1 Label (Available: 500 pieces) → Can make 500 bottles

Result: Can make 30 bottles (limited by bottle availability)
```

## Benefits

1. ✅ **Accurate** - Stock reflects actual production capability
2. ✅ **Automatic** - No manual stock updates needed
3. ✅ **Real-time** - Always shows current production capacity
4. ✅ **Logical** - Matches real-world manufacturing process
5. ✅ **Less Error-Prone** - No risk of manual stock entry mistakes

## User Workflow

### Before:
1. Add raw materials with quantities
2. Create product
3. **Manually enter product stock** ❌
4. **Manually update stock** when materials change ❌

### After:
1. Add raw materials with quantities
2. Create product with raw material requirements
3. ✅ **System automatically calculates production capacity**
4. ✅ **Stock updates automatically** when materials change

## Views Behavior

### Products Create/Edit:
- Shows: Name, SKU, Price, Description
- Shows message: "Stock availability is calculated automatically based on raw materials"
- Edit form shows: "Current production capacity: X units"

### Products Index:
| Column | Display |
|--------|---------|
| Name | Product name |
| SKU | Product SKU |
| Price | Product price |
| Can Make | X units (calculated) |
| Raw Materials | Count of materials |
| Status | Available / Insufficient Materials |

### Stock Analysis:
- Shows calculated "Can Make" values for all products
- Shows detailed raw material breakdown in modal
- Identifies products that can't be made

### Dashboard:
- Shows "Products That Can't Be Made" (0 units)
- Lists products with insufficient materials
- Helps identify procurement needs

## Migration Rollback

If needed, you can rollback the migration:
```bash
php artisan migrate:rollback
```

This will restore the `current_stock` and `min_stock_level` columns to the products table.

## Testing

After these changes, test:
1. ✅ Create a new product - should not ask for stock
2. ✅ Edit existing product - should show calculated capacity
3. ✅ View products list - should show "Can Make" column
4. ✅ Check dashboard - should show unavailable products
5. ✅ Add raw material stock - product capacity should update automatically

## Notes

- Database columns removed permanently
- No data loss - stock is calculated from raw materials
- Backward compatible migration included (can rollback if needed)
- All references to manual stock removed from codebase
