<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\Product;
use App\Models\RawMaterial;

class ProductController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        $products = Product::with('rawMaterials')->paginate(15);
        return view('products.index', compact('products'));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        $rawMaterials = RawMaterial::where('is_active', true)->get();
        return view('products.create', compact('rawMaterials'));
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $request->validate([
            'name' => 'required|string|max:255',
            'sku' => 'nullable|string|max:255|unique:products',
            'price' => 'required|numeric|min:0',
            'description' => 'nullable|string',
            'raw_materials' => 'array',
            'raw_materials.*.raw_material_id' => 'required|exists:raw_materials,id',
            'raw_materials.*.quantity_required' => 'required|numeric|min:0.01'
        ]);

        $product = Product::create([
            'name' => $request->name,
            'sku' => $request->sku,
            'price' => $request->price,
            'description' => $request->description,
            'is_active' => true
        ]);

        // Attach raw materials
        if ($request->has('raw_materials')) {
            foreach ($request->raw_materials as $rawMaterial) {
                $product->rawMaterials()->attach($rawMaterial['raw_material_id'], [
                    'quantity_required' => $rawMaterial['quantity_required']
                ]);
            }
        }

        return redirect()->route('products.index')->with('success', 'Product created successfully');
    }

    /**
     * Display the specified resource.
     */
    public function show(Product $product)
    {
        $product->load('rawMaterials');
        return view('products.show', compact('product'));
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(Product $product)
    {
        $rawMaterials = RawMaterial::where('is_active', true)->get();
        $product->load('rawMaterials');
        return view('products.edit', compact('product', 'rawMaterials'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, Product $product)
    {
        $request->validate([
            'name' => 'required|string|max:255',
            'sku' => 'nullable|string|max:255|unique:products,sku,' . $product->id,
            'price' => 'required|numeric|min:0',
            'description' => 'nullable|string',
            'raw_materials' => 'array',
            'raw_materials.*.raw_material_id' => 'required|exists:raw_materials,id',
            'raw_materials.*.quantity_required' => 'required|numeric|min:0.01'
        ]);

        $product->update([
            'name' => $request->name,
            'sku' => $request->sku,
            'price' => $request->price,
            'description' => $request->description
        ]);

        // Sync raw materials
        if ($request->has('raw_materials')) {
            $syncData = [];
            foreach ($request->raw_materials as $rawMaterial) {
                $syncData[$rawMaterial['raw_material_id']] = [
                    'quantity_required' => $rawMaterial['quantity_required']
                ];
            }
            $product->rawMaterials()->sync($syncData);
        } else {
            $product->rawMaterials()->detach();
        }

        return redirect()->route('products.index')->with('success', 'Product updated successfully');
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(Product $product)
    {
        $product->delete();
        return redirect()->route('products.index')->with('success', 'Product deleted successfully');
    }
}
