<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsToMany;
use Illuminate\Database\Eloquent\Relations\HasMany;

class Product extends Model
{
    protected $fillable = [
        'name',
        'description',
        'sku',
        'price',
        'is_active'
    ];

    protected $casts = [
        'price' => 'decimal:2',
        'is_active' => 'boolean'
    ];

    public function rawMaterials(): BelongsToMany
    {
        return $this->belongsToMany(RawMaterial::class, 'product_raw_materials')
                    ->withPivot('quantity_required')
                    ->withTimestamps();
    }

    public function orderItems(): HasMany
    {
        return $this->hasMany(OrderItem::class);
    }

    public function productRawMaterials(): HasMany
    {
        return $this->hasMany(ProductRawMaterial::class);
    }

    /**
     * Calculate how many units of this product can be made with current raw material stock
     */
    public function calculateCanMake(): int
    {
        if ($this->rawMaterials->count() === 0) {
            return 0;
        }

        $canMake = PHP_INT_MAX;
        
        foreach ($this->rawMaterials as $rawMaterial) {
            $required = $rawMaterial->pivot->quantity_required;
            $available = $rawMaterial->current_stock;
            $possible = floor($available / $required);
            $canMake = min($canMake, $possible);
        }

        return $canMake;
    }

    /**
     * Check if product can be made with current raw material stock
     */
    public function canMake(): bool
    {
        return $this->calculateCanMake() > 0;
    }
}
