<?php

namespace App\Services;

use App\Models\Product;
use App\Models\RawMaterial;
use App\Models\Order;
use App\Models\OrderItem;
use Illuminate\Support\Facades\DB;

class InventoryService
{
    /**
     * Calculate raw material requirements for a given order
     */
    public function calculateRawMaterialRequirements(Order $order): array
    {
        $requirements = [];
        
        foreach ($order->orderItems as $orderItem) {
            $product = $orderItem->product;
            $quantity = $orderItem->quantity;
            
            foreach ($product->rawMaterials as $rawMaterial) {
                $requiredQuantity = $rawMaterial->pivot->quantity_required * $quantity;
                
                if (!isset($requirements[$rawMaterial->id])) {
                    $requirements[$rawMaterial->id] = [
                        'raw_material' => $rawMaterial,
                        'total_required' => 0
                    ];
                }
                
                $requirements[$rawMaterial->id]['total_required'] += $requiredQuantity;
            }
        }
        
        return $requirements;
    }
    
    /**
     * Update raw material stock after order processing
     */
    public function updateRawMaterialStock(Order $order): void
    {
        $requirements = $this->calculateRawMaterialRequirements($order);
        
        DB::transaction(function () use ($requirements) {
            foreach ($requirements as $rawMaterialId => $requirement) {
                $rawMaterial = RawMaterial::find($rawMaterialId);
                $rawMaterial->current_stock -= $requirement['total_required'];
                $rawMaterial->save();
            }
        });
    }
    
    /**
     * Check if there's enough raw material stock for an order
     */
    public function checkStockAvailability(Order $order): array
    {
        $requirements = $this->calculateRawMaterialRequirements($order);
        $shortages = [];
        
        foreach ($requirements as $rawMaterialId => $requirement) {
            $rawMaterial = $requirement['raw_material'];
            $available = $rawMaterial->current_stock;
            $required = $requirement['total_required'];
            
            if ($available < $required) {
                $shortages[] = [
                    'raw_material' => $rawMaterial,
                    'required' => $required,
                    'available' => $available,
                    'shortage' => $required - $available
                ];
            }
        }
        
        return $shortages;
    }
    
    /**
     * Get inventory status for all raw materials
     */
    public function getInventoryStatus(): array
    {
        return RawMaterial::where('is_active', true)
            ->get()
            ->map(function ($rawMaterial) {
                return [
                    'raw_material' => $rawMaterial,
                    'status' => $rawMaterial->current_stock <= $rawMaterial->min_stock_level ? 'low' : 'ok',
                    'percentage' => $rawMaterial->min_stock_level > 0 
                        ? min(100, ($rawMaterial->current_stock / $rawMaterial->min_stock_level) * 100)
                        : 100
                ];
            })
            ->toArray();
    }
    
    /**
     * Add raw material stock
     */
    public function addRawMaterialStock(int $rawMaterialId, float $quantity): void
    {
        $rawMaterial = RawMaterial::findOrFail($rawMaterialId);
        $rawMaterial->current_stock += $quantity;
        $rawMaterial->save();
    }
    
    /**
     * Get products that can't be made (insufficient raw materials)
     */
    public function getUnavailableProducts(): array
    {
        return Product::where('is_active', true)
            ->with('rawMaterials')
            ->get()
            ->filter(function($product) {
                return $product->calculateCanMake() === 0;
            })
            ->map(function($product) {
                return [
                    'id' => $product->id,
                    'name' => $product->name,
                    'sku' => $product->sku,
                    'can_make' => $product->calculateCanMake()
                ];
            })
            ->toArray();
    }
}